﻿//=============================================================================
//
//          Copyright (c) 2022 Beijing Dreamagic Science and Technology Co.,Ltd.
//                          All Rights Reserved.
//
//=============================================================================
using System;
using System.Runtime.InteropServices;
using static Unity.XR.Qiyu.QiyuXRCorePlugin;
using DataPtr = System.IntPtr;
using FunPtr = System.IntPtr;

namespace Unity.XR.Qiyu
{
    public static partial class QiyuXRCore
    {
        public static void Update(float deltaTime)
        {
            if (!QiyuPlatform.IsAndroid)
                return;

            QiyuXRCorePlugin.QVR_Update(deltaTime);
        }

        public static void OnApplicationPause(bool b)
        {
            if (!QiyuPlatform.IsAndroid)
                return;

            QiyuXRCorePlugin.QVR_OnApplicationPause(b);
        }

        /// <summary>
        /// 获取地面高度
        /// </summary>
        public static float GetFloorLevel()
        {
            if (!QiyuPlatform.IsAndroid)
                return 0;

            return QiyuXRCorePlugin.QVR_GetFloorLevel();
        }

        /// <summary>
        /// 获取用户瞳距设置
        /// </summary>
        public static float GetUserIPD()
        {
            if (!QiyuPlatform.IsAndroid)
                return 0;

            return QiyuXRCorePlugin.QVR_GetUserIPD() * 1.0f / 1000;
        }

        /// <summary>
        /// 设置3dof模式，1:3dof,0:6dof
        /// </summary>
        /// <returns></returns>
        public static void SetAppTrackingMode(int mode)
        {
            if (!QiyuPlatform.IsAndroid)
                return;

            QiyuXRCorePlugin.QVR_SetAppTrackingMode(mode);
        }

        /// <summary>
        /// 设置屏幕刷新率
        /// </summary>
        /// <param name="refreshRate">刷新率</param>
        /// <returns></returns>
        public static bool TrySetDisplayRefreshRate(float refreshRate)
        {
            if (!QiyuPlatform.IsAndroid)
                return false;

            return QiyuXRCorePlugin.QVR_SetDisplayRefreshRate(refreshRate);
        }

        /// <summary>
        /// 获取屏幕刷新率
        /// </summary>
        /// <param name="refreshRate">返回值</param>
        /// <returns></returns>
        public static bool TryGetDisplayRefreshRate(out float refreshRate)
        {
            if (!QiyuPlatform.IsAndroid)
            {
                refreshRate = 72;
                return true;
            }

            float rate = 0;
            if (QiyuXRCorePlugin.QVR_GetDisplayRefreshRate(ref rate))
            {
                refreshRate = rate;
                return true;
            }
            else
            {
                refreshRate = rate;
                return false;
            }
        }

        private static QiyuXRCorePlugin.NativeBuffer cachedRefreshRatesBuffer = new QiyuXRCorePlugin.NativeBuffer(0);
        private static float[] cachedRefreshRatesManagedBuffer = new float[0];
        /// <summary>
        /// 获取可用刷新率列表
        /// </summary>
        /// <param name="refreshRates">返回列表</param>
        /// <returns></returns>
        public static bool TryGetAvailableDisplayRefreshRates(out float[] refreshRates)
        {
            if (!QiyuPlatform.IsAndroid)
            {
                refreshRates = new float[] { 72, 90 };
                return true;
            }

            int count = 0;
            QiyuXRCorePlugin.QVR_GetAvailableDisplayRefreshRates(IntPtr.Zero, ref count);
            if (count > 0)
            {
                if (cachedRefreshRatesBuffer.GetCapacity() < count)
                    cachedRefreshRatesBuffer.Reset(count);

                if (cachedRefreshRatesManagedBuffer.Length < count)
                    cachedRefreshRatesManagedBuffer = new float[count];

                QiyuXRCorePlugin.QVR_GetAvailableDisplayRefreshRates(cachedRefreshRatesBuffer.GetPointer(), ref count);

                Marshal.Copy(cachedRefreshRatesBuffer.GetPointer(), cachedRefreshRatesManagedBuffer, 0, count);

                refreshRates = new float[count];
                for (int i = 0; i < count; i++)
                {
                    refreshRates[i] = cachedRefreshRatesManagedBuffer[i];

                }
                return true;
            }
            else
            {
                refreshRates = new float[0];
                return false;
            }
        }

        public struct QiyuDeviceInfo
        {
            public string Name_HMD;
            public string Name_CtrL;
            public string Name_CtrR;
            public string Manufacturer;
            public string SN_HMD;
            public string SN_CtrL;
            public string SN_CtrR;
        };

        /// <summary>
        /// 获取设备信息
        /// </summary>
        /// <returns>设备信息</returns>
        public static QiyuDeviceInfo GetDeviceInfo()
        {
            if (!QiyuPlatform.IsAndroid)
                return new QiyuDeviceInfo();

            QiyuDeviceInfoNative data = QiyuXRCorePlugin.QVR_GetDeviceInfo();
            QiyuDeviceInfo info = new QiyuDeviceInfo();
            info.Name_HMD = Marshal.PtrToStringAuto(data.Name_HMD);
            info.Name_CtrL = Marshal.PtrToStringAuto(data.Name_CtrL);
            info.Name_CtrR = Marshal.PtrToStringAuto(data.Name_CtrR);
            info.Manufacturer = Marshal.PtrToStringAuto(data.Manufacturer);
            info.SN_HMD = Marshal.PtrToStringAuto(data.SN_HMD);
            info.SN_CtrL = Marshal.PtrToStringAuto(data.SN_CtrL);
            info.SN_CtrR = Marshal.PtrToStringAuto(data.SN_CtrR);

            return info;
        }

        /// <summary>
        /// 开启SeeThrough
        /// </summary>
        /// <param name="visible"></param>
        public static void EnableSeeThrough(bool visible)
        {
            if (QiyuPlatform.IsAndroid)
            {
                QiyuXRCorePlugin.QVR_EnableSeeThrough(visible);
            }
        }
        private static Action<int> seethrough_callback = null;
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void SeeThroughCallBack(int state);
        public static void RegisterSeeThroughCallback(Action<int> callback)
        {
            if (seethrough_callback == null)
            {
                seethrough_callback = callback;
                if (QiyuPlatform.IsAndroid)
                {
                    SeeThroughCallBack callback_delegate = OnSeeThroughChanged;
                    QiyuXRCorePlugin.QVR_SetSeeThroughCallback(Marshal.GetFunctionPointerForDelegate(callback_delegate));
                }
            }
            else
            {
                seethrough_callback += callback;
                UnityEngine.Debug.Log("User RegisterSeeThroughCallback!");
            }
        }

        public static void UnRegisterSeeThroughCallback(Action<int> callback)
        {
            if (seethrough_callback != null)
            {
                seethrough_callback -= callback;
                UnityEngine.Debug.Log("User UnRegisterSeeThroughCallback!");
            }
        }

        [AOT.MonoPInvokeCallback(typeof(SeeThroughCallBack))]
        public static void OnSeeThroughChanged(int state)
        {
            if (seethrough_callback != null)
            {
                seethrough_callback.Invoke(state);
            }
        }

        public static void DisableSxrRender(bool visible)
        {
            if (QiyuPlatform.IsAndroid)
            {
                QiyuXRCorePlugin.QVR_DisableSxrRender(visible);
            }
        }
    }
}
