﻿//=============================================================================
//
//          Copyright (c) 2022 Beijing Dreamagic Science and Technology Co.,Ltd.
//                          All Rights Reserved.
//
//=============================================================================
using AOT;
using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Runtime.InteropServices;
using UnityEditor;
using UnityEngine;
using Newtonsoft.Json;
using System.Threading;
using Unity.XR.Qiyu;

public class PreviewSettings
{
    public int connectType;
    public string ip;
    public int port;
    public int fps;
    public int eyebufferSize;
}

[InitializeOnLoad]
public class QiyuDirectPreviewManager
{
    public const string QiyuDirectPreviewPath = "QiyuDirectPreview";

    private static void PlaymodeStateChangedEvent(PlayModeStateChange state)
    {
        if (state == PlayModeStateChange.EnteredPlayMode)
        {
            RegisterDebugCallback_Core(OnDebugCallback);
        }
        else
        {
            RegisterDebugCallback_Core(null);
        }
    }

    static QiyuDirectPreviewManager()
    {
#if UNITY_EDITOR
        EditorApplication.playModeStateChanged += PlaymodeStateChangedEvent;
#endif
    }
    string TAG = "QiyuDirectPreviewManager";

    public string directViewApkPath = Path.GetFullPath("Packages/com.unity.xr.qiyu/Assets/DirectPreview/QiyuPreview-release.apk");

    private static QiyuDirectPreviewManager _me = new QiyuDirectPreviewManager();

    public static QiyuDirectPreviewManager GetMe()
    {
        return _me;
    }

    public string deviceCachePath
    {
        get => "/sdcard/QiyuConfig/";
    }

    public void InstallDirectPreviewApk()
    {
        QiyuDirectPreviewEditorWindow.PrintLog("installing ...", QiyuDirectPreviewEditorWindow.LogType.Normal);
        if (!File.Exists(directViewApkPath))
        {
            QiyuDirectPreviewEditorWindow.PrintLog(string.Format("can not find {0}", directViewApkPath), QiyuDirectPreviewEditorWindow.LogType.Error);
        }
        string[] pushCommand = { "-d install", directViewApkPath };
        QiyuADBTools.GetMe().RunCommandAsync(pushCommand, (s, e) =>
        {
            QiyuDirectPreviewEditorWindow.PrintLog("install app success.", QiyuDirectPreviewEditorWindow.LogType.Success);
        });
    }

    public void UninstallDirectPreviewApk()
    {
        QiyuDirectPreviewEditorWindow.PrintLog("uninstalling ...", QiyuDirectPreviewEditorWindow.LogType.Normal);

        string[] pushCommand = { "-d uninstall", "com.qiyu.PreviewAgent" };
        QiyuADBTools.GetMe().RunCommandAsync(pushCommand, (s, e) =>
        {
            QiyuDirectPreviewEditorWindow.PrintLog("uninstall app success.", QiyuDirectPreviewEditorWindow.LogType.Success);
        });
    }

    public void StartDirectPreviewApp()
    {
        string[] pushCommand = { "-d shell am ", "start -n com.qiyu.PreviewAgent/com.unity3d.player.UnityPlayerActivity" };
        string info, err;
        if (QiyuADBTools.GetMe().RunCommand(pushCommand, null, out info, out err) != 0)
        {
            QiyuDirectPreviewEditorWindow.PrintLog(string.Format("start app error {0}", err), QiyuDirectPreviewEditorWindow.LogType.Error);
        }
        if (string.IsNullOrEmpty(err))
        {
            QiyuDirectPreviewEditorWindow.PrintLog("start app success", QiyuDirectPreviewEditorWindow.LogType.Success);
        }
        else
        {
            if (err.Contains("Activity class {com.qiyu.PreviewAgent/com.unity3d.player.UnityPlayerActivity} does not exist"))
            {
                QiyuDirectPreviewEditorWindow.PrintLog("No agent detected, please install QiyuPreview Agent first.", QiyuDirectPreviewEditorWindow.LogType.Error);
            }
            else
            {
                QiyuDirectPreviewEditorWindow.PrintLog(string.Format("start app error {0}", err), QiyuDirectPreviewEditorWindow.LogType.Error);
            }
        }

    }

    public bool StopDirectPreviewApp()
    {
        if (!QiyuADBTools.GetMe().IsAdbAvailable())
        {
            return false;
        }
        string output, error;
        string[] appStartCommand = { "-d shell", "am force-stop  com.qiyu.PreviewAgent" };
        if (QiyuADBTools.GetMe().RunCommand(appStartCommand, null, out output, out error) == 0)
        {
            return true;
        }
        return false;
    }

    public bool PushPreviewConfig(PreviewSettings settings)
    {
        string path = Path.Combine("QiyuDirectPreviewPath");
        string fileName = "preview_config.json";
        string configFile = Path.Combine(path, fileName);
        if (!Directory.Exists(path))
        {
            Directory.CreateDirectory(path);
        }
        string config = JsonConvert.SerializeObject(settings);

        StreamWriter writer = new StreamWriter(configFile, false);
        writer.Write(config);
        writer.Close();

        string configFilePath = Path.Combine(Path.GetFullPath(configFile));
        string[] pushCommand = { "-d push", "\"" + configFilePath + "\"", "\"" + deviceCachePath + "\"" };
        string output, error;

        string[] lsCommand = { "-d shell ls ", deviceCachePath };

        if (QiyuADBTools.GetMe().RunCommand(lsCommand, null, out output, out error) != 0)
        {
            if (output.Contains("No such file or directory") || error.Contains("No such file or directory"))
            {
                string[] mkdirCommand = { "-d shell", "mkdir -p", "\"" + deviceCachePath + "\"" };
                if (QiyuADBTools.GetMe().RunCommand(mkdirCommand, null, out output, out error) != 0)
                {
                    Debug.LogError(output);
                    QiyuDirectPreviewEditorWindow.PrintLog(string.IsNullOrEmpty(error) ? output : error, QiyuDirectPreviewEditorWindow.LogType.Error);
                    return false;
                }
            }
        }
        if (QiyuADBTools.GetMe().RunCommand(pushCommand, null, out output, out error) != 0)
        {
            Debug.LogError(output);
            QiyuDirectPreviewEditorWindow.PrintLog(string.IsNullOrEmpty(error) ? output : error, QiyuDirectPreviewEditorWindow.LogType.Error);
            return false;
        }
        return true;
    }

    /// <summary>
    /// 
    /// </summary>
    /// <param name="isUSB"></param>
    /// <param name="fps"></param>
    /// <param name="eyebufferSize"></param>
    /// <param name="connectType">0:USB,1:Wifi</param>
    public void ConnectDevice(PreviewSettings config)
    {
        if (!Application.isPlaying)
        {
            QiyuDirectPreviewEditorWindow.PrintLog("Please Play Scene", QiyuDirectPreviewEditorWindow.LogType.Error);
            return;
        }
        StopDirectPreviewApp();
        Thread.Sleep(1);
        StartDirectPreviewApp();
        if (config.connectType == 0)
        {
            string[] pushCommand = { "-d forward tcp:" + config.port.ToString(), "tcp:" + config.port.ToString() };
            string output, error;
            if (QiyuADBTools.GetMe().RunCommand(pushCommand, null, out output, out error) != 0)
            {
                QiyuDirectPreviewEditorWindow.PrintLog(string.IsNullOrEmpty(error) ? output : error, QiyuDirectPreviewEditorWindow.LogType.Error);
                return;
            }
        }
        if (!QiyuLoader.isQiyuLoaderInitialize)
        {
            QiyuDirectPreviewEditorWindow.PrintLog("please select qiyu sdk in XR Plug-in Management for windows.", QiyuDirectPreviewEditorWindow.LogType.Error);
            return;
        }
        if (!QiyuPlugin_ConnectDirectPreview(config.ip, config.port, config.fps, config.eyebufferSize))
        {
            QiyuDirectPreviewEditorWindow.PrintLog("please select qiyu sdk in XR Plug-in Management for windows.", QiyuDirectPreviewEditorWindow.LogType.Error);
        }
    }

    #region QiyuXRCore
    //------------------------------------------------------------------------------------------------


    [DllImport("QiyuXRCore", CallingConvention = CallingConvention.Cdecl)]
    static extern void RegisterDebugCallback_Core(debugCallback cb);

    //Create string param callback delegate
    delegate void debugCallback(IntPtr request, int color, int size);
    enum Color { red, green, blue, black, white, yellow, orange };
    [MonoPInvokeCallback(typeof(debugCallback))]
    static void OnDebugCallback(IntPtr request, int color, int size)
    {
        try
        {
            //Ptr to string
            string debug_string = Marshal.PtrToStringAnsi(request, size);

            if (string.IsNullOrEmpty(debug_string))
                return;

            //Add Specified Color
            debug_string =
                String.Format("{0}{1}{2}{3}{4}",
                "<color=",
                ((Color)color).ToString(),
                ">",
                debug_string,
                "</color>"
                );
            UnityEngine.Debug.Log(debug_string);
        }
        catch (Exception ex)
        {
            UnityEngine.Debug.LogError(ex.Message);
        }
    }

    //TODO:临时代码
    [DllImport("QiyuXRCore", CallingConvention = CallingConvention.Cdecl)]
    public static extern void QVR_ConnectClient(string ip, int port);

    [DllImport("QiyuXRCore", CallingConvention = CallingConvention.Cdecl)]
    public static extern float QVR_GetUserIPD();

    [DllImport("QiyuXRCore", CallingConvention = CallingConvention.Cdecl)]
    public static extern void QVR_Start_PC();

    [DllImport("QiyuXRCore", CallingConvention = CallingConvention.Cdecl)]
    public static extern void QVR_OnDestroy_PC();

    [DllImport("QiyuXRCore", CallingConvention = CallingConvention.Cdecl)]
    public static extern void QVR_Update_PC(float deltaTime);

    #endregion

    #region UnityQiyuVR

    [DllImport("UnityQiyuVR", CallingConvention = CallingConvention.Cdecl)]
    public static extern bool QiyuPlugin_ConnectDirectPreview(string ip, int port, int fps, int eyebufferSize);


    #endregion
}
